"use client";

import { use, useEffect, useState } from "react";
import { notFound } from "next/navigation";
import wordpressAPI from "@/lib/wordpress";
import { useLanguage } from "@/contexts/LanguageContext";

interface PageProps {
  params: Promise<{ slug: string }>; // ✅ params is now a Promise
}

export default function PortfolioPage({ params }: PageProps) {
  const { slug } = use(params); // ✅ unwrap the Promise
  const { language } = useLanguage();
  const [portfolio, setPortfolio] = useState<any | null>(null);

  useEffect(() => {
    const fetchPortfolio = async () => {
      try {
        const data = await wordpressAPI.getPostBySlugAndType("portfolio", slug, language);
        setPortfolio(data);
      } catch (error) {
        console.error("Error loading portfolio:", error);
        setPortfolio(null);
      }
    };

    fetchPortfolio();
  }, [slug, language]);

  // Still fetching → show nothing or skeleton
  if (portfolio === null) {
    return null;
  }

  // If nothing found
  if (!portfolio) {
    notFound();
  }

  return (
    <div className="portfolio-content">
      <div className="container my-8">
        <h1
          className="text-4xl font-bold mb-8 text-gray-900 dark:text-gray-100"
          dangerouslySetInnerHTML={{
            __html:
              typeof portfolio.title === "string"
                ? portfolio.title
                : portfolio.title?.rendered || "",
          }}
        />
        <div
          className="te-portfolio-post prose dark:prose-invert prose-md prose-img:rounded-xl prose-a:text-primary-600 dark:prose-a:text-primary-400 hover:prose-a:text-gray-500 max-w-none"
          dangerouslySetInnerHTML={{
            __html:
              typeof portfolio.content === "string"
                ? portfolio.content
                : portfolio.content?.rendered || "",
          }}
        />
      </div>
    </div>
  );
}
